"use client";
import { useEffect, useState } from 'react';
import { useParams } from 'next/navigation';
import { getSupabaseClient } from '@/config/supabase';
import type { Tables } from '@/types/supabase';
import Image from 'next/image';
import ProductDetailSkeleton from '@/components/products/ProductDetailSkeleton';
import Link from 'next/link';
import toast from 'react-hot-toast';
import ProductDetailActions from '@/components/products/ProductDetailActions';

export default function ProductDetailPage() {
  const { id } = useParams<{ id: string }>();
  const [product, setProduct] = useState<Tables['content_items'] | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    setLoading(true);
    setError(null);
    getSupabaseClient()
      .from('content_items')
      .select('*')
      .eq('id', id)
      .single()
      .then(({ data, error }) => {
        if (error || !data) {
          setError('Product not found.');
          toast.error('Product not found.');
        } else {
          setProduct(data);
        }
      })
      .catch(() => {
        setError('Unable to fetch product.');
        toast.error('Unable to fetch product.');
      })
      .finally(() => setLoading(false));
  }, [id]);

  if (loading) return <ProductDetailSkeleton />;
  if (error)
    return (
      <div className="text-red-600 text-center my-16 text-lg min-h-[300px]">
        {error}
      </div>
    );
  if (!product) return null;

  return (
    <section className="max-w-5xl mx-auto px-4 py-12 flex flex-col md:flex-row gap-10">
      <div className="w-full md:w-[400px] flex-shrink-0">
        <Image
          src={product.image_url}
          alt={product.title}
          width={400}
          height={400}
          className="rounded-3xl shadow-glass object-cover border border-muted"
        />
      </div>
      <div className="flex-1 flex flex-col gap-6">
        <h1 className="text-3xl font-heading font-bold mb-2">{product.title}</h1>
        <p className="text-muted-foreground text-lg mb-4">{product.description}</p>
        <div className="flex items-center gap-4 mb-3">
          <span className="text-2xl font-bold text-primary">${product.price.toFixed(2)}</span>
          <span className="px-3 py-1 rounded bg-accent text-accent-foreground text-xs font-semibold uppercase">
            {product.category}
          </span>
        </div>
        <ProductDetailActions product={product} />
        <Link href="/" className="px-6 py-2 rounded-lg bg-muted text-foreground font-semibold border border-muted hover:bg-background transition-colors">
          Continue Shopping
        </Link>
      </div>
    </section>
  );
}
